using System;
using System.Collections;
using System.IO;
using DynaPDF;

namespace examples
{
	class Class1
	{
      static int PDFError(IntPtr Data, int ErrCode, IntPtr ErrMessage, int ErrType)
      {
         Console.Write(string.Format("{0}", System.Runtime.InteropServices.Marshal.PtrToStringAnsi(ErrMessage)));
         return 0; // We try to continue if an error occurrs. Any other return value breaks processing.
      }

      /*
         Incremental updates are useful if small changes should be saved quickly, e.g. when adding an annotation or reply to one,
         adding form fields and so on.

         This example is a bit simplified since we know that the file contains only one annotation and a handle of an annotation
         is just an array index from 0 though annotation count -1.
      */

      static byte[] CreateTestFile(CPDF PDF)
      {
         PDF.CreateNewPDF(null);

         PDF.SetPageCoords(TPageCoord.pcTopDown);

         PDF.Append();
            PDF.SquareAnnot(50.0, 50.0, 200.0, 100.0, 3.0, CPDF.NO_COLOR, 255, TPDFColorSpace.csDeviceRGB, "Jim", "Test", "Just test...");
         PDF.EndPage();

         if (!PDF.CloseFile()) return null;
         // This function creates a copy of the buffer and this is exactly what we need in this example.
         byte[] buffer = PDF.GetBuffer();

         // We can release the original buffer. This also resets the PDF instance,
         // just as CloseFile() would do if the file would be created on a local drive.
         PDF.FreePDF();
         return buffer;
      }

      static bool LoadTestFile(CPDF PDF, byte[] Buffer)
      {
         PDF.CreateNewPDF(null);
         // This flag also sets the flags ifImportAsPage | ifImportAll, and if2UseProxy | if2CopyEncryptDict to make sure that
         // anything is imported and nothing gets changed.
         PDF.SetImportFlags2(TImportFlags2.if2IncrementalUpd);
         if (PDF.OpenImportBuffer(Buffer, TPwdType.ptOpen, null) < 0) return false;
         return (PDF.ImportPDFFile(1, 1.0, 1.0) > 0);
      }

      static byte[] SaveFile(CPDF PDF)
      {
         if (!PDF.CloseFile()) return null;
         byte[] retval = PDF.GetBuffer();
         // We can release the original buffer. This also resets the PDF instance,
         // just as CloseFile() would do if the file would be created on a local drive.
         PDF.FreePDF();
         return retval;
      }


		[STAThread]
		static void Main(string[] args)
		{
		   try
		   {
            int reply;
            byte[] buffer;
            CPDF pdf = new CPDF();
            pdf.SetOnErrorProc(IntPtr.Zero, new TErrorProc(PDFError));

            // We create the base file in memory in this example
            buffer = CreateTestFile(pdf);

            // CreateTest() created a copy of the PDF buffer. We are now the owner of the buffer.
            // The PDF instance has now it's inital state with the exception that the font cache, if already loaded, stays in memory.
            if (LoadTestFile(pdf, buffer))
            {
               // We don't need the buffer anymore
               buffer = null;
               // We do not search for the annotation here since we know that the file contains only one annotation
               // and the handle of this annotation is zero since this is just an array index.
               reply = pdf.SetAnnotMigrationState(0, TAnnotState.asCreateReply, "Harry");
               pdf.SetAnnotString(reply, TAnnotString.asContent, "Hi Jim, your test annotation looks fine!");
               buffer = SaveFile(pdf);
               if (LoadTestFile(pdf, buffer))
               {
                  buffer = null;

                  reply = pdf.SetAnnotMigrationState(reply, TAnnotState.asCreateReply, "Tommy");
                  pdf.SetAnnotString(reply, TAnnotString.asContent, "Just a test whether I can reply to a reply...");
                  buffer = SaveFile(pdf);
                  if (LoadTestFile(pdf, buffer))
                  {
                     buffer = null;

                     reply = pdf.SetAnnotMigrationState(reply, TAnnotState.asCreateReply, "Jim");
                     pdf.SetAnnotString(reply, TAnnotString.asContent, "Seems to work very well!");
                     // No fatal error occurred?
                     if (pdf.HaveOpenDoc())
                     {
                        // We write the output file into the current directory.
                        String filePath = Directory.GetCurrentDirectory() + "\\out.pdf";
                        // OK, now we can open the output file.
                        if (!pdf.OpenOutputFile(filePath)) return;
                        if (pdf.CloseFile())
                        {
                           Console.Write("PDF file \"{0}\" successfully created!\n", filePath);
                           System.Diagnostics.Process p = new System.Diagnostics.Process();
                           p.StartInfo.FileName = filePath;
                           p.Start();
                        }
                     }
                  }
               }
            }
         }catch(Exception e)
         {
            Console.Write(e.Message);
            Console.Read();
         }
		}
	}
}
